/*
 * Routines for dealing with the fabric deltas
 */

#include <sys/time.h>

#include "db.h"

#include "libfma.h"
#include "lf_fabric.h"
#include "lf_fabric_db.h"
#include "lf_scheduler.h"

#include "fms.h"
#include "fms_error.h"
#include "fms_fabric.h"
#include "fms_fabric_delta.h"
#include "fms_notify.h"


/*
 * Local prototypes
 */
static void fms_do_db_flush(void *v);
void
fms_note_new_host(
  struct lf_host *hp)
{
  FMS_HOST(hp)->update_state = FMS_UPDATE_ADD;
}

/*
 * A host has changed - mark it unless it is new
 */
void
fms_note_changed_host(
  struct lf_host *hp)
{
  /* as long as this host is not new, mark it as changed */
  if (FMS_HOST(hp)->update_state != FMS_UPDATE_ADD) {
    FMS_HOST(hp)->update_state = FMS_UPDATE_CHANGE;
  }
}

/*
 * Mark a host as needing to be flushed
 */
void
fms_commit_host(
  struct lf_host *hp)
{
  FMS_HOST(hp)->commit = TRUE;
  fms_flush_db_changes();
}

/*
 * Flush any changes for this host to the database.
 * This is pretty brute force - if changing, we eliminate any references
 * to this host from the database.  Then, we re-add all necessary entries
 * for this host.
 */
void
fms_flush_host(
  struct lf_host *hp)
{
  fms_notify(FMS_EVENT_INFO, "Committing changes for host %s", hp->hostname);

  if (FMS_HOST(hp)->update_state == FMS_UPDATE_CHANGE
      || FMS_HOST(hp)->update_state == FMS_UPDATE_REMOVE) {
    fms_remove_host_from_db(hp);
  }

  if (FMS_HOST(hp)->update_state == FMS_UPDATE_CHANGE
      || FMS_HOST(hp)->update_state == FMS_UPDATE_ADD) {
    fms_add_host_to_db(hp);
  }

  FMS_HOST(hp)->update_state = FMS_UPDATE_NONE;
  FMS_HOST(hp)->commit = FALSE;
}

void
fms_flush_all_hosts(
  struct lf_fabric *fp)
{
  int h;

  for (h=0; h<fp->num_hosts; ++h) {
    struct lf_host *hp;

    hp = fp->hosts[h];
    if (FMS_HOST(hp)->commit) {
      fms_flush_host(hp);
    }
  }
}

/*
 * Add this host to the DB fabric and the DB host table
 */
void
fms_add_host_to_db(
  struct lf_host *hp)
{
#ifdef CLONE
  fms_add_host_to_db_fabric(hp);
#endif
  fms_add_host_to_db_table(hp);

  fms_flush_db_changes();
}

/*
 * To "update" a host, just remove and re-add it
 */
void
fms_update_host_in_db(
  struct lf_host *hp)
{
  fms_remove_host_from_db(hp);
  fms_add_host_to_db(hp);
}

/*
 * Remove a host from the fabric database
 */
void
fms_remove_host_from_db(
  struct lf_host *hp)
{
  fms_remove_host_from_db_table(hp);
#ifdef CLONE
  fms_remove_host_from_db_fabric(hp);
#endif

  fms_flush_db_changes();
}

/*
 * Add a linecard to the database table and fabric.
 */
void
fms_add_linecard_to_db(
  struct lf_linecard *lp)
{
#ifdef CLONE
  fms_add_linecard_to_db_fabric(lp);
#endif
  fms_add_linecard_to_db_table(lp);

  fms_flush_db_changes();
}

/*
 * Add an enclosure to the database.
 */
void
fms_add_enclosure_to_db(
  struct lf_enclosure *ep)
{
#ifdef CLONE
  fms_add_enclosure_to_db_fabric(ep);
#endif
  fms_add_enclosure_to_db_table(ep);

  fms_flush_db_changes();
}

/*
 * Remove an enclosure from the database
 */
void
fms_remove_enclosure_from_db(
  struct lf_enclosure *ep)
{
  fms_remove_enclosure_from_db_table(ep);
  fms_flush_db_changes();
}

/*
 * Add this link to the DB fabric and the DB
 */
void
fms_add_topo_link_to_db(
  union lf_node *np,
  int port)
{
  /* Add to the database fabric */
#ifdef CLONE
  fms_add_topo_link_to_db_fabric(np, port);
#endif

  /* Add to the actual database */
  fms_add_topo_link_to_db_table(np, port);

  fms_flush_db_changes();
}

/*
 * Remove this link from the DB and DB fabric
 */
void
fms_remove_topo_link_from_db(
  union lf_node *np,
  int port)
{
  /* Remove this from the DB fabric */
#ifdef CLONE
  fms_remove_topo_link_from_db_fabric(np, port);
#endif

  fms_remove_topo_link_from_db_table(np, port);

  fms_flush_db_changes();
}

/*
 * If there is not already a database flush scheduled, schedule one
 */
void
fms_flush_db_changes()
{
  if (F.fabvars->db_flush_task == NULL) {
    F.fabvars->db_flush_task =
      lf_schedule_event(fms_do_db_flush, NULL, FMS_DB_FLUSH_DELAY);
    if  (F.fabvars->db_flush_task == NULL) {
      LF_ERROR(("Error schdeuling DB flush"));
    }
  }
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Perform a flush of the database out to disk
 */
static void
fms_do_db_flush(
  void *v)
{
  struct lf_fabric_db *fdp;
  int rc;

  fms_notify(FMS_EVENT_DEBUG, "Flushing database");

  /* re-arm ability to flush */
  F.fabvars->db_flush_task = NULL;

  /* write all host changes to the database */
  fms_flush_all_hosts(F.fabvars->fabric);

  /* write the database tables */
  fdp = F.fabvars->fabric_db;

  /* flush the DB */
  rc = lf_flush_fabric_db(fdp);
  if (rc != 0) LF_ERROR(("Error flushing fabric DB"));

  return;

 except:
  fms_perror_exit(1);
}

/*
 * Add this link the the DB unless the other end is bogus
 */
void
fms_add_topo_link_to_db_table(
  union lf_node *np,
  int port)
{
  struct lf_fabric_db *fdp;
  union lf_node *onp;
  int rc;

  fdp = F.fabvars->fabric_db;

  /* If this end is bogus, skip it */
  if (fms_is_bogus(np)) {
    return;
  }

  onp = lf_follow_topo_link(np, port, NULL);

  /* If no other node, no link to add */
  if (onp == NULL) {
    return;
  }

  /* If other end bogus, skip it */
  if (fms_is_bogus(onp)) {
    return;
  }

  /* Add the link and return status */
  rc = lf_add_topo_link_to_db(fdp, np, port, LF_DS_NEW);
  if (rc != 0) {
    LF_ERROR(("Error adding topo link to DB"));
  }
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Add a host to the database, and all items underneath it.
 * Then, add any links for the NICs in this host.
 */
void
fms_add_host_to_db_table(
  struct lf_host *hp)
{
  struct lf_fabric_db *fdp;
  int rc;
  int n;

  fdp = F.fabvars->fabric_db;

  /* Do not add bogus host to database */
  if (FMS_HOST(hp)->is_bogus) {
    return;
  }

  /* Add the host itself */
  rc = lf_add_host_to_db(fdp, hp);
  if (rc != 0) LF_ERROR(("Error adding host to DB"));

  /* Add each NIC in this host */
  for (n=0; n<hp->num_nics; ++n) {
    struct lf_nic *nicp;
    int p;

    nicp = hp->nics[n];
    rc = lf_add_nic_to_db(fdp, nicp);
    if (rc != 0) LF_ERROR(("Error adding NIC to DB"));

    /* add the links for this NIC */
    for (p=0; p<nicp->num_ports; ++p) {
      fms_add_topo_link_to_db(LF_NODE(nicp), p);
    }
  }
  return;

 except:
  fms_perror_exit(1);
}

#ifdef CLONE
/*
 * Make a copy of this host in the DB fabric
 */
void
fms_add_host_to_db_fabric(
  struct lf_host *ohp)
{
  struct lf_host *nhp;
  int rc;

  /* create a cloned host (with NICs) */
  nhp = lf_clone_host(ohp);
  if (nhp == NULL) LF_ERROR(("Error adding host to fabric"));

  /* add a new host struct in the DB fabric */
  rc = lf_add_existing_host_to_fabric(F.fabvars->db_fabric, nhp);
  if (rc != 0) LF_ERROR(("Error adding host to fabric"));

  /* clone all the links */
  rc = lf_clone_host_links(F.fabvars->fabric, ohp, F.fabvars->db_fabric, nhp);
  if (rc != 0) LF_ERROR(("Error cloning host links"));

  return;

 except:
  fms_perror_exit(1);
}
#endif

/*
 * Remove a host from the fabric database
 */
void
fms_remove_host_from_db_table(
  struct lf_host *hp)
{
  struct lf_fabric_db *fdp;
  int rc;

  fdp = F.fabvars->fabric_db;

  rc = lf_remove_link_from_db(fdp, hp->hostname, -1, -1, -1);
  if (rc != 0) LF_ERROR(("Error removing links from DB"));

  rc = lf_remove_nic_from_db(fdp, hp->hostname, -1);
  if (rc != 0) LF_ERROR(("Error removing NICs from DB"));

  rc = lf_remove_host_from_db(fdp, hp->hostname);
  if (rc != 0) LF_ERROR(("Error removing host from DB"));

  return;

 except:
  fms_perror_exit(1);
}

#ifdef CLONE
/*
 * Remove a host from the DB fabric
 */
void
fms_remove_host_from_db_fabric(
  struct lf_host *hp)
{
  struct lf_fabric *dfp;
  struct lf_host *chp;

  /* get DB fabric and clone host */
  dfp = F.fabvars->db_fabric;
  chp = FMS_HOST(hp)->clone_mate;

  /* remove this from the DB fabric */
  lf_remove_host_from_fabric(dfp, chp);

  /* XXX - what about lingering links? */

  /* Free the cloned host struct */
  lf_free_host(chp);
}
#endif

/*
 * Add an enclosure to the database.
 */
void
fms_add_enclosure_to_db_table(
  struct lf_enclosure *ep)
{
  struct lf_fabric_db *fdp;
  int s;
  int rc;
  
  /* don't add bogus enclosures */
  if (FMS_ENC(ep)->is_bogus) {
    return;
  }

  fms_notify(FMS_EVENT_DEBUG, "Adding enclosure %s to DB", ep->name);

  fdp = F.fabvars->fabric_db;

  /* Add the enclosure */
  rc = lf_add_enclosure_to_db(fdp, ep);
  if (rc != 0) LF_ERROR(("Error adding enclosure to DB"));

  /* Add each linecard to the DB */
  for (s=0; s<ep->num_lc_slots; ++s) {
    struct lf_linecard *lp;

    lp = ep->slots[s];
    if (lp != NULL) {
      fms_add_linecard_to_db_table(lp);
    }
  }

  return;

 except:
  fms_perror_exit(1);
}

/*
 * Remove an enclosure from the fabric database
 */
void
fms_remove_enclosure_from_db_table(
  struct lf_enclosure *ep)
{
  struct lf_fabric_db *fdp;
  int rc;

  fdp = F.fabvars->fabric_db;

  /* remove all the links to this enclosure */
  rc = lf_remove_link_from_db(fdp, ep->name, -1, -1, -1);
  if (rc != 0) LF_ERROR(("Error removing enclosure links from DB"));

  /* remove all enclosures */
  rc = lf_remove_linecard_from_db(fdp, ep->name, -1);
  if (rc != 0) LF_ERROR(("Error removing linecards from DB"));

  /* lastly, remove the enclosure itself */
  rc = lf_remove_enclosure_from_db(fdp, ep->name);
  if (rc != 0) LF_ERROR(("Error removing enclosure from DB"));

  return;

 except:
  fms_perror_exit(1);
}

#ifdef CLONE
/*
 * Make a copy of this enclosure in the DB fabric
 */
void
fms_add_enclosure_to_db_fabric(
  struct lf_enclosure *ep)
{
  struct lf_enclosure *cep;
  struct lf_fabric *dfp;
  int rc;

  /* create a cloned copy of the enclosure */
  cep = lf_clone_enclosure(ep);
  if (cep == NULL) {
    LF_ERROR(("Error cloning enclosure %s", ep->name));
  }

  /* Add to the DB fabric */
  dfp = F.fabvars->db_fabric;
  rc = lf_add_existing_enclosure_to_fabric(dfp, cep);
  if (rc != 0) {
    LF_ERROR(("Error adding %s to DB fabric", cep->name));
  }

  return;

 except:
  fms_perror_exit(1);
}

/*
 * Remove the cloned copy of this enclosure from DB fabric
 */
void
fms_remove_enclosure_from_db_fabric(
  char *name)
{
  struct lf_enclosure *cep;
  struct lf_fabric *dfp;

  /* get pointer to DB fabric */
  dfp = F.fabvars->db_fabric;

  /* Find the clone enclosure */
  cep = lf_find_enclosure_by_name(dfp, name);
  if (cep == NULL) {
    LF_ERROR(("Error finding clone enclosure for %s", name));
  }

  /* remove this from the DB fabric */
  fms_remove_enclosure_from_fabric(dfp, cep);

  return;

 except:
  fms_perror_exit(1);
}
#endif

/*
 * Add a linecard to the database
 */
void
fms_add_linecard_to_db_table(
  struct lf_linecard *lp)
{
  struct lf_fabric_db *fdp;
  int xc;
  int rc;

  /* Don't add things belonging to bogus enclosures */
  if (FMS_ENC(lp->enclosure)->is_bogus) {
    return;
  }

  fms_notify(FMS_EVENT_DEBUG, "Adding linecard %s:%d to DB",
               lp->enclosure->name, lf_slot_display_no(lp));

  fdp = F.fabvars->fabric_db;

  /* Add the linecard */
  rc = lf_add_linecard_to_db(fdp, lp);
  if (rc != 0) LF_ERROR(("Error adding linecard to DB"));

  /* Add any links for this linecard */
  for (xc=0; xc<lp->num_xcvrs; ++xc) {
    struct lf_xcvr *xcp;
    int p;

    xcp = LF_XCVR(lp->xcvrs[xc]);
    for (p=0; p<xcp->num_conns; ++p) {
      union lf_node *onp;

      onp = xcp->ports[p];
      if (onp != NULL) {
	rc = lf_add_phys_link_to_db(fdp, LF_NODE(xcp), p, LF_DS_NEW);
	if (rc != 0) LF_ERROR(("Error adding link to DB"));
      }
    }
  }
  return;

 except:
  fms_perror_exit(1);
}

#ifdef CLONE
/*
 * Make a copy of this linecard in the DB fabric
 */
void
fms_add_linecard_to_db_fabric(
  struct lf_linecard *lp)
{
  struct lf_enclosure *cep;
  struct lf_linecard *clp;

  /* Find matching cloned enclosure */
  cep = FMS_ENC(lp->enclosure)->clone_mate;

  /* create a cloned linecard (with xbars) */
  clp = lf_clone_linecard(cep, lp);
  if (clp == NULL) LF_ERROR(("Error adding linecard to fabric"));

  return;

 except:
  fms_perror_exit(1);
}

/*
 * Make this link in the DB fabric
 */
void
fms_add_topo_link_to_db_fabric(
  union lf_node *np,
  int port)
{
  struct lf_fabric *dfp;
  union lf_node *cnp;
  union lf_node *onp;
  union lf_node *conp;
  int oport;

  /* get to the DB fabric */
  dfp = F.fabvars->db_fabric;

  /* follow the link */
  onp = lf_follow_topo_link(np, port, &oport);
  if (onp == NULL) {
    LF_ERROR(("No link to clone!"));
  }

  /* find these nodes in the DB fabric */
  cnp = lf_find_clone_counterpart(np, dfp);
  if (cnp == NULL) {
    LF_ERROR(("Cannot find clone partner"));
  }
  conp = lf_find_clone_counterpart(onp, dfp);
  if (conp == NULL) {
    LF_ERROR(("Cannot find other clone partner"));
  }

  /* Link the nodes */
  lf_connect_topo_nodes(cnp, port, conp, oport);
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Break this link in the DB fabric
 */
void
fms_remove_topo_link_from_db_fabric(
  union lf_node *np,
  int port)
{
  struct lf_fabric *dfp;
  union lf_node *cnp;

  /* get to the DB fabric */
  dfp = F.fabvars->db_fabric;

  /* find this node in the DB fabric */
  cnp = lf_find_clone_counterpart(np, dfp);
  if (cnp == NULL) {
    LF_ERROR(("Cannot find clone partner"));
  }

  /* remove the link */
  lf_remove_link_by_topo(cnp, port);

  return;

 except:
  fms_perror_exit(1);
}
#endif

/*
 * Remove this link from the DB
 * This only needs to be done for non-internal links.  Internal links
 * are characterized by the physical link being the same as the topological
 * link.
 */
void
fms_remove_topo_link_from_db_table(
  union lf_node *np,
  int port)
{
  char *name;
  int slot;
  int dbport;
  int subport;
  struct lf_fabric_db *fdp;
  int rc;

  fdp = F.fabvars->fabric_db;

  /* get the DB values for this link */
  rc = lf_get_topo_link_db_info(np, port, &name, &slot, &dbport, &subport);
  if (rc == -1) {
    LF_ERROR(("Error getting DB link info"));
  }

  /* If name is NULL, this link does not go in the database (implicit link) */
  if (name != NULL) {

    /* remove this link */
    rc = lf_remove_link_from_db(fdp, name, slot, dbport, subport);
    if (rc != 0) LF_ERROR(("Error removing link from from DB"));
  }

  return;

 except:
  fms_perror_exit(1);
}
